﻿using System;
using System.Collections;
using System.Resources;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;
using System.Collections.Specialized;
using System.Reflection;
using System.Text;
using VeteransAffairs.Registries.BusinessManager;
using System.Collections.Generic;
using VeteransAffairs.Registries.Business;
using System.Web.Configuration;

/// <summary>
/// Summary description for BasePage
/// </summary>
public abstract class BasePage : Page
{
    public Menu MenuHorizontalMenu;
    public Menu MenuLeftNavigation;
    public Label LabelBreadCrumb;
    public string _redirectAfterSave;

    protected static ResourceManager resources;

    private bool _errorOnSavedReceived;
    private Dictionary<string, string> _OnStartupAlertErrorCollection;

    public BasePage() 
    {
        resources = new ResourceManager("Portal.strings", Assembly.GetExecutingAssembly());
        Load += new EventHandler(BasePage_Load);
    }

    public BasePage(bool errorOnSavedReceived)
    {
        _errorOnSavedReceived = errorOnSavedReceived;
    }

    protected UserAccountManager CurrentUser
    {
        get
        {
            return (UserAccountManager)HttpContext.Current.User;
        }
    }

    public Dictionary<string, string> OnStartupAlertErrorCollection
    {
        get
        {
            if (_OnStartupAlertErrorCollection == null)
                _OnStartupAlertErrorCollection = new Dictionary<string, string>();

            return _OnStartupAlertErrorCollection;
        }
    }

    public bool IgnoreCapabilityCheck { get; set; }
    public bool ReadOnly { get; set; }
    public bool BusinessObjectSaveEventReceived { get; set; }
    public bool SuppressSaveSuccessMessage { get; set; }
    public bool IsSaveSuccess { get; set; }

    protected IReadOnlyList<string> NavigationSources { get; set; }


    protected void manager_BOSaveSuccess(object sender, BaseBO.BOSaveSuccessEventArgs e)
    {
        if (!_errorOnSavedReceived)
        {
            BusinessObjectSaveEventReceived = true;//these two flags are reduntant and should be replaced with e.IsSaveSuccess
        }
    }
    
    protected void BuildMenusAndBreadCrumb()
    {
        //Get menu list based on user access
        string[] pagesAccessibleByUser = ((UserAccountManager)HttpContext.Current.User).GetPagesByCode();

        MenuHorizontalMenu = (Menu)Master.FindControl("MenuHorizontalMenu");
        MenuLeftNavigation = (Menu)Master.FindControl("MenuLeftNavigation");
        LabelBreadCrumb = (Label)Master.FindControl("LabelBreadCrumb");

        //Populate left menu and breadcrumb - based on user roles
        MenuBuilder.BuildDynamicNavigationItems(HttpContext.Current.Request.Url.PathAndQuery, MenuHorizontalMenu, TopMenuName(), MenuLeftNavigation, MenuItemName(), LabelBreadCrumb, pagesAccessibleByUser);
    }

    protected void MakeControlsReadOnly(ControlCollection controls)
    {
        foreach (Control c in controls)
        {
            if (c is TextBox || c is RadioButton ||
                c is DropDownList || c is CheckBox || c is RadioButtonList || c is CheckBoxList)
            {
                PropertyInfo propertyInfo = c.GetType().GetProperty("Enabled");
                propertyInfo.SetValue(c, false, null);
            }

            if (c.HasControls())
            {
                MakeControlsReadOnly(c.Controls);
            }
        }
    }

    protected void MakeControlsWriteable(ControlCollection controls)
    {
        foreach (Control c in controls)
        {
            if (c is TextBox || c is RadioButton ||
                c is DropDownList || c is CheckBox || c is RadioButtonList || c is CheckBoxList)
            {
                PropertyInfo propertyInfo = c.GetType().GetProperty("Enabled");
                propertyInfo.SetValue(c, true, null);
            }

            if (c.HasControls())
            {
                MakeControlsWriteable(c.Controls);
            }
        }
    }

    protected T GetOrStore<T>(string key, Func<T, bool> shouldUpdate, Func<T> getObject)
        where T : class
    {
        T result = Session[key] as T;

        if (result == default(T) || shouldUpdate(result))
        {
            result = getObject();
            Session[key] = result;
        }

        return result;
    }

    protected void SetReferringPage()
    {
        var referrer = ViewState["referrer"];
        if (referrer == null)
            ViewState["referrer"] = Request.UrlReferrer;
    }

    protected void RedirectToReferringPage()
    {
        Uri from = ViewState["referrer"] as Uri;
        if (UriUtility.IsValidPathSource(NavigationSources, from))
            Response.Redirect(from.PathAndQuery);
    }

    /// <summary>
    /// Similar to SetReferringPage() but uses Session vs ViewState to save page originally navigated in on;
    /// useful if ViewState is not sufficient to get the original referring page
    /// </summary>
    /// <param name="pageReferrerName">ex. "BioLabInterpretation_Referrer": BioLabInterpretation save originally referring page </param>
    protected void SetOriginalReferringPage(string pageReferrerName)
    {
        var navigatedFrom = SessionItemReference<Uri>.Get(Session[pageReferrerName]);
        if (navigatedFrom == null)
            Session[pageReferrerName] = SessionItemFactory.Create(Request.UrlReferrer);
    }

    /// <summary>
    /// Similar to RedirectToReferringPage() but uses original referring page, in case page has multiple redirects;
    /// validates referrals but redirects to original referring page
    /// </summary>
    /// <param name="pageReferrerName">ex. "BioLabInterpretation_Referrer": BioLabInterpretation save originally referring page </param>
    protected void RedirectToOriginalReferringPage(string pageReferrerName)
    {
        Uri from = ViewState["referrer"] as Uri;
        if (UriUtility.IsValidPathSource(NavigationSources, from))
        {
            var navigatedFrom = SessionItemReference<Uri>.Get(Session[pageReferrerName]);
            if (navigatedFrom != null)
            {
                Session.Remove(pageReferrerName);
                Response.Redirect(navigatedFrom.PathAndQuery);
            }
        }
    }

    /// <summary>
    /// TopMenuName is the menu item in the top horizontal menu where this page falls under
    /// </summary>
    public abstract string TopMenuName();

    //to do
    //need to spend more time on design
    //menu items should be database driven
    //menu item is mapped to combination of page and business object
    public abstract string MenuItemName();

    /// <summary>
    /// The list of capabilities that a user needs to access this page.  A single page may have more than one
    /// capability associated with it.
    /// </summary>
    public abstract string AppObjectCode();

    /// <summary>
    /// Check if the current user has EFR_TEFSC_COORDINATOR role 
    /// </summary>
    /// <returns></returns>
    public bool IsCurrentUserSuperUser()
    {
       return CurrentUser.GetRoles().Contains((int)RoleCode.EFR_TEFSC_COORDINATOR);
    }

    /// <summary>
    /// Select returns the base application path.
    /// </summary>
    /// <param name="context">Context object</param>
    /// <returns>Returns the base application path.</returns>
    public static string RootPath(HttpContext context)
    {
        string urlSuffix = context.Request.Url.Authority + context.Request.ApplicationPath;
        return context.Request.Url.Scheme + @"://" + urlSuffix + "/";
    }
    /// <summary>
    /// You must pass in a string that uses the QueryStringHelper.DELIMITER as the delimiter.
    /// This will also append the "?" to the beginning of the query string.
    /// </summary>
    /// <param name="queryString"></param>
    /// <returns></returns>
    public static string EncryptQueryString(string queryString)
    {
        //return StringHelpers.EncryptQueryString(queryString);
        string qs = "";
        return qs;
    }
    public static NameValueCollection DecryptQueryString(string queryString)
    {
        NameValueCollection myColl = new NameValueCollection();
        myColl.Add("a", "1");
        return myColl;
    }
    public static string EncryptQueryString(NameValueCollection queryString)
    {        
        string myString = "";
        return myString;
    }
    public string ConvertDateToString(DateTime? date)
    {
        if (!date.HasValue)
            return "";
        else
            return date.GetValueOrDefault().ToShortDateString();
    }
    public void PrepareGridViewForExport(Control gv)
    {
        Hashtable htControls = new Hashtable();

        htControls.Add("LinkButton", "Text");
        htControls.Add("HyperLink", "Text");
        htControls.Add("DropDownList", "SelectedItem");
        htControls.Add("CheckBox", "Checked");

        Literal l = new Literal();

        for (int i = 0; i < gv.Controls.Count; i++)
        {
            if ((null != htControls[gv.Controls[i].GetType().Name]) || 
                (null != htControls[gv.Controls[i].GetType().BaseType.Name]))
            {
                l.Text = GetControlPropertyValue(gv.Controls[i], htControls);
                gv.Controls.Remove(gv.Controls[i]);
                gv.Controls.AddAt(i, l);
            }

            if (gv.Controls[i].HasControls())
            {
                PrepareGridViewForExport(gv.Controls[i]);
            }
        }
    }
    private string GetControlPropertyValue(Control control, Hashtable htControls)
    {
        Type controlType = control.GetType();
        string strControlType = controlType.Name;
        string strReturn = "Error";
        bool bReturn;
        PropertyInfo[] ctrlProps = controlType.GetProperties();
        string ExcelPropertyName = (string)htControls[strControlType];

        if (ExcelPropertyName == null)
        {
            ExcelPropertyName = (string)htControls[control.GetType().BaseType.Name];
            if (ExcelPropertyName == null)
                return strReturn;
        }

        foreach (PropertyInfo ctrlProp in ctrlProps)
        {
            if (ctrlProp.Name == ExcelPropertyName && ctrlProp.PropertyType == typeof(string))
            {
                strReturn = (string)ctrlProp.GetValue(control, null);
                break;
            }

            if (ctrlProp.Name == ExcelPropertyName &&
                    ctrlProp.PropertyType == typeof(bool))
            {
                bReturn = (bool)ctrlProp.GetValue(control, null);
                strReturn = bReturn ? "True" : "False";
                break;
            }

            if (ctrlProp.Name == ExcelPropertyName && ctrlProp.PropertyType == typeof(ListItem))
            {
                strReturn = ((ListItem)(ctrlProp.GetValue(control, null))).Text;
                break;
            }
        }

        return strReturn;
    }

    public void ExportGridViewToExcel(GridView GridView1, string filename)
    {
        PrepareGridViewForExport(GridView1);
        HtmlForm form = new HtmlForm();
        string attachment = "attachment; filename=" + filename;
        Response.ClearContent();
        Response.AddHeader("content-disposition", attachment);
        Response.ContentType = "application/vnd.ms-excel";
        System.IO.StringWriter sw = new System.IO.StringWriter();
        HtmlTextWriter htw = new HtmlTextWriter(sw);
        form.Controls.Add(GridView1);
        Controls.Add(form);
        form.RenderControl(htw);
        Response.Write(sw.ToString());
        Response.End();
    }
    
    //Chapter 9: Printing
    public string GetPrintButtonScript(Button btn)
    {        
        StringBuilder printButtonScript = new StringBuilder();

        //Get the postback script.
        string postback = Page.ClientScript.GetPostBackEventReference(btn, "");

        //Change target to a new window.  Name the window the current date and time so multiple windows can
        //be opened.
        printButtonScript.Append("var today = new Date();");
        printButtonScript.Append("var newWindowName = today.getFullYear().toString() + today.getMonth().toString() + today.getDate().toString() + today.getHours().toString() + today.getHours().toString() + today.getMinutes().toString() + today.getSeconds().toString() + today.getMilliseconds().toString();");

        printButtonScript.Append("document.forms[0].target = newWindowName;");

        //TODO: Added root path after this was turned in.
        //Show the please wait screen.
        //todo: printButtonScript.Append("window.open('" + RootPath(base.Context) + "/Reports/PleaseWait.html', newWindowName, 'scrollbars=yes,status=no,resizable=yes');");

        //Add the postback script.
        printButtonScript.Append(postback + ";");

        //Reset target back to itself so other controls will post back to this form.
        printButtonScript.Append("document.forms[0].target='_self';");
        //Return false to stop page submitting.
        printButtonScript.Append("return false;" + Environment.NewLine);

        return printButtonScript.ToString();
    }

    protected override void OnPreInit(EventArgs e)
    {
       Theme = "EFR";
        base.OnPreInit(e);
    }

    protected override void OnLoad(EventArgs e)
    {
        base.OnLoad(e);
    }

    protected override void OnInit(EventArgs e)
    {
        if (DisplayApplicationWarning() == false)
        {
            Session["SkipWarning"] = SessionItemFactory.Create("Yes");

            base.OnInit(e);
            CheckCapabilities();

            if (!isQueryStringValid())
            {
                //redirect to error handling page
                Response.Redirect("ErrorPage.aspx?id=qs");
            }

            if (!IsPostBack)
            {
                BuildMenusAndBreadCrumb();
            }
        }
    }
    protected override void OnPreRender(EventArgs e)
    {
        
        RenderStartUpMessage(BusinessObjectSaveEventReceived);
        
        base.OnPreRender(e);
    }

    /// <summary>
    /// Get whether or not the page requires authentication to access.  Assume true but allow specific pages to override.
    /// </summary>
    protected virtual bool RequiresAuthentication
    {
        get { return true; }
    }
    public virtual void CheckCapabilities()
    {
        if (IgnoreCapabilityCheck == false)
        {            
            //Get user permissions
            if (CurrentUser.HasAccessToRegistry() == false)
                NoAccessToPage();
            else
            {
                Permissions Perm = CurrentUser.GetPermissions(AppObjectCode());
                if (Perm.NoAccess == true)
                    NoAccessToPage();
                else if (Perm.ReadOnly == true)
                    MakeFormReadOnly(AppObjectCode(), this.Controls);
            }
        }
    }

    protected virtual void NoAccessToPage()
    {     
        //You do not access to the system.
        //If a page has more than one capability you should override this method because a user could
        //have access to one section but not another so you do not want them to get an error
        Response.Redirect("ErrorPage.aspx?id=100");
        //throw new AccessViolationException("You do not have access to this screen.");
    }

    public virtual void CustomReadOnlyLogic(string capabilityName)
    {
        //Override this method in a page that has custom logic for non standard controls on the screen.
    }

    /// <summary>
    /// The default implementation will make all controls disabled.
    /// If you have more than one capability associated with a page you should override this method
    /// with the special logic for each capability in the page.
    /// </summary>
    /// <param name="capabilityName"></param>
    /// <param name="controls"></param>
    public virtual void MakeFormReadOnly(string capabilityName, ControlCollection controls)
    {
        ReadOnly = true;
        MakeControlsReadOnly(controls);
        CustomReadOnlyLogic(capabilityName);
    }

    private bool isQueryStringValid()
    {
        //this iterates over querysting parameters
        //this will handle known querystring parameters and validate range, if any not valid range throw exception
        //if unknown parameter is found, throw exception

        bool _isValid = true;

        NameObjectCollectionBase.KeysCollection qryString = Request.QueryString.Keys ;
        
        foreach (string qryStringKey in qryString)
        {
            switch (qryStringKey)
            {
                case "id":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;

                case "dodFragmentId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;

                case "dodFragmentLabId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;

                case "dodFragmentAnalyteId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;

                case "fragmentId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;

                case "statusId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "searchText":
                    _isValid = Helpers.isValidString(Request.QueryString[qryStringKey]);
                    break;
                case "OrderType":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "ActivityId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "KitTypeId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "InstitutionId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "referralId":                   
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "showOriginal":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                
                case "activityId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;

                case "patientId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "CaseId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "contactId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "intActivityId":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "RptType":
                    _isValid = Helpers.isValidInteger(Request.QueryString[qryStringKey]);
                    break;
                case "strDate":
                    _isValid = Helpers.isValidString(Request.QueryString[qryStringKey]);
                    break;
                case "from":
                    _isValid = Helpers.isValidString(Request.QueryString[qryStringKey]);
                    break;
                case "patientImageId":
                    _isValid = Helpers.isValidString(Request.QueryString[qryStringKey]);
                    break;
                case "searchFirstName":
                    _isValid = Helpers.isValidString(Request.QueryString[qryStringKey]);
                    break;
                case "searchLastName":
                    _isValid = Helpers.isValidString(Request.QueryString[qryStringKey]);
                    break;
                case "searchSsn":
                    _isValid = Helpers.isValidString(Request.QueryString[qryStringKey]);
                    break;
                case "searchIcn":
                    _isValid = Helpers.isValidString(Request.QueryString[qryStringKey]);
                    break;
                default:
                    _isValid = false;   
                    break;
            }

            if (!_isValid) 
            {
                Logging.WriteLogToFile(User.Identity.Name, "Invalid querystring parameter value for: " + qryStringKey.ToString());
                Logging.WriteLogToFile(User.Identity.Name, "Full URL: " + Request.RawUrl.ToString());
                break;
            }
        }

        return _isValid;
    }
    private void BasePage_Load(object sender, EventArgs e)
    {
        // Kick the user out if the page requires authentication but the user somehow isn't (like if the auth system has been
        // maliciously bypassed).
        if (RequiresAuthentication && (!User.Identity.IsAuthenticated))
        {
            Response.Redirect("ErrorPage.aspx?id=404");
        }
    }

    private string CreateStartUpMessage(bool isSaveSuccessMessage)
    {

        string Message = string.Empty;

        if (isSaveSuccessMessage && _OnStartupAlertErrorCollection == null && !SuppressSaveSuccessMessage)
        {
            Message = "Save Successful.";
        }
        else
        {
            if (_OnStartupAlertErrorCollection != null)
            {
                foreach (KeyValuePair<string, string> item in _OnStartupAlertErrorCollection)
                    Message += item.Value + "\\n";
            }

        }

        return Message;
  
    }
    /// <summary>
    /// Renders the massage in javascript alert. Modify if needed for section 508 compliance
    /// </summary>
    private void RenderStartUpMessage(bool IsSaveSuccessMessage)
    {
        StringBuilder sb = new StringBuilder();
        string Message = CreateStartUpMessage(IsSaveSuccessMessage);

        if (Message != string.Empty)
        {
            HiddenField HiddenFieldShowSaveMessage = this.Master.FindControl("HiddenFieldShowSaveMessage") as HiddenField;
            if (HiddenFieldShowSaveMessage != null)
            {
                HiddenFieldShowSaveMessage.Value = "1";
            }

            HiddenField HiddenFieldSaveMessage = this.Master.FindControl("HiddenFieldSaveMessage") as HiddenField;
            if (HiddenFieldSaveMessage != null)
            {
                HiddenFieldSaveMessage.Value = Message;
            }

            HiddenField HiddenFieldRedirectOnSave = this.Master.FindControl("HiddenFieldRedirectOnSave") as HiddenField;
            if (HiddenFieldRedirectOnSave != null)
            {
                HiddenFieldRedirectOnSave.Value = _redirectAfterSave;
            }
        }
    }

    private bool DisplayApplicationWarning()
    {
        bool result = false;

        if (TopMenuName().ToLower() == "administration")
        {
            result = false;
        }
        else if (Session["SkipWarning"] != null)
        {
            if (SessionItemValue.Get(Session["SkipWarning"]) == "Yes")
            {
                result = false;
            }
        }

        else
        {
            RegistriesCommonManager commonManager = new RegistriesCommonManager();

            APPLICATION_STATUS appStatus = commonManager.GetApplicationStatus(WebConfigurationManager.AppSettings.Get("Registry"));

            if (appStatus != null)
            {
                if (appStatus.PROCESS_FLAG > 0)
                {
                    result = true;

                    Session["warningMessage"] = SessionItemFactory.Create(appStatus.MESSAGE);
                    Session["theUrl"] = SessionItemFactory.Create(Request.Url.Segments[Request.Url.Segments.Length - 1] + Request.Url.Query);

                    Response.Redirect("TermsConsent.aspx?id=" + appStatus.PROCESS_FLAG);
                }
            }
        }

        return result;
    }
}